const taskList = document.getElementById("taskList");
const form = document.getElementById("taskForm");
const messageDiv = document.getElementById("message");
const addBtn = document.getElementById("addBtn");
const cancelBtn = document.getElementById("cancelUpdate");

let isEditing = false;
let editingTaskId = null;

// Load all tasks
async function loadTasks() {
    try {
        const res = await fetch("/tasks");
        const tasks = await res.json();
        
        taskList.innerHTML = '';
        
        if (tasks.length === 0) {
            taskList.innerHTML = '<li style="text-align:center; color: #999;">No tasks yet. Add one above!</li>';
            return;
        }
        
        tasks.forEach(task => {
            const li = document.createElement("li");
            
            li.innerHTML = `
                <div>
                    <strong>${task.title}</strong><br>
                    <small>${task.description}</small><br>
                    <small>Due: ${task.dueDate || 'No due date'}</small>
                </div>
                <div class="task-buttons">
                    <span class="status ${task.status}">${task.status}</span>
                    <button class="editBtn" onclick="editTask(${task.id})">Edit</button>
                    <button class="delBtn" onclick="deleteTask(${task.id})">Delete</button>
                </div>
            `;
            
            taskList.appendChild(li);
        });
    } catch (error) {
        showMessage("Error loading tasks", false);
    }
}

// Show message
function showMessage(msg, isSuccess = true) {
    messageDiv.textContent = msg;
    messageDiv.style.display = "block";
    messageDiv.style.backgroundColor = isSuccess ? "#4caf50" : "#f44336";
    setTimeout(() => {
        messageDiv.style.display = "none";
    }, 3000);
}

// Create or Update task
form.addEventListener("submit", async (e) => {
    e.preventDefault();

    const title = document.getElementById("title").value;
    const description = document.getElementById("description").value;
    const dueDate = document.getElementById("dueDate").value;
    const status = document.getElementById("status").value;

    if (!title || !description) {
        showMessage("Title and description cannot be empty", false);
        return;
    }

    if (isEditing) {
        // Update existing task
        const res = await fetch(`/tasks/${editingTaskId}`, {
            method: "PUT",
            headers: { "Content-Type": "application/json" },
            body: JSON.stringify({ title, description, dueDate, status })
        });

        const data = await res.json();
        showMessage(data.message);
        
        // Reset editing mode
        isEditing = false;
        editingTaskId = null;
        addBtn.textContent = "Add Task";
        cancelBtn.style.display = "none";
    } else {
        // Create new task
        const res = await fetch("/tasks", {
            method: "POST",
            headers: { "Content-Type": "application/json" },
            body: JSON.stringify({ title, description, dueDate, status })
        });

        const data = await res.json();

        if (res.ok) {
            showMessage(data.message);
        } else {
            showMessage(data.message, false);
        }
    }

    form.reset();
    loadTasks();
});

// Edit task
async function editTask(id) {
    try {
        const res = await fetch(`/tasks/${id}`);
        const task = await res.json();

        document.getElementById("title").value = task.title;
        document.getElementById("description").value = task.description;
        document.getElementById("dueDate").value = task.dueDate || '';
        document.getElementById("status").value = task.status;

        isEditing = true;
        editingTaskId = id;
        addBtn.textContent = "Update Task";
        cancelBtn.style.display = "inline-block";

        window.scrollTo(0, 0);
    } catch (error) {
        showMessage("Error loading task for editing", false);
    }
}

// Cancel editing
cancelBtn.addEventListener("click", () => {
    form.reset();
    isEditing = false;
    editingTaskId = null;
    addBtn.textContent = "Add Task";
    cancelBtn.style.display = "none";
});

// Delete task
async function deleteTask(id) {
    if (!confirm("Are you sure you want to delete this task?")) {
        return;
    }
    
    const res = await fetch(`/tasks/${id}`, { method: "DELETE" });
    const data = await res.json();
    showMessage(data.message);
    loadTasks();
}

// Initial load
loadTasks();